package edu.uky.ai.lp;

import java.util.function.Predicate;

/**
 * Represents the current state of a game of "Hunt the Wumpus."
 * 
 * @author Stephen G. Ware
 */
public class Game {
	
	/** The possible contents of each square */
	private enum Square { WUMPUS, PIT, GOLD }
	
	/** The dungeon map */
	private final Square[][] map = new Square[4][4];
	
	/** A way to track which squares are visited */
	private final boolean[][] visited = new boolean[4][4];
	
	/** The player's current file (a through d) */
	private int playerFile = -1;
	
	/** The player's current rank (1 through 4) */
	private int playerRank = -1;
	
	/** Records whether or not the player has collected the goal */
	private boolean hasGold = false;
	
	/** Records whether or not the game has ended */
	private boolean over = false;
	
	/**
	 * <p>Constructs a new game instance for a given dungeon map.  The map
	 * should be given as an array of strings, starting with square A4 and
	 * ending with square D1.</p>
	 * <p>The following string values indicate:</p>
	 * <ul>
	 * <li><tt>PL</tt> = the player</li>
	 * <li><tt>WM</tt> = the wumpus</li>
	 * <li><tt>PT</tt> = a pit</li>
	 * <li><tt>GD</tt> = the gold</li>
	 * </ul>
	 * <p>All other indices should be null.</p>
	 * 
	 * @param map the map, as described above
	 */
	public Game(String[] map) {
		int index = 0;
		for(int rank=3; rank>=0; rank--) {
			for(int file=0; file<4; file++) {
				String square = map[index];
				index++;
				if(square == null)
					continue;
				else if(square.equalsIgnoreCase("PL")) {
					playerFile = file;
					playerRank = rank;
					visited[file][rank] = true;
				}
				else if(square.equalsIgnoreCase("WM"))
					this.map[file][rank] = Square.WUMPUS;
				else if(square.equalsIgnoreCase("PT"))
					this.map[file][rank] = Square.PIT;
				else if(square.equalsIgnoreCase("GD"))
					this.map[file][rank] = Square.GOLD;
			}
		}
	}
	
	/**
	 * Applies a predicate to all squares adjacent to a given square.  This
	 * method short-circuits, so the first time the predicate returns true,
	 * this method will return true without testing other squares.
	 * 
	 * @param file the file of the given square
	 * @param rank the rank of the given square
	 * @param predicate the predicate to be tested
	 * @return true if the predicate returned true for any square, false otherwise
	 */
	private final boolean adjacent(int file, int rank, Predicate<Square> predicate) {
		if(file < 3 && predicate.test(map[file + 1][rank]))
			return true;
		if(file > 0 && predicate.test(map[file - 1][rank]))
			return true;
		if(rank < 3 && predicate.test(map[file][rank + 1]))
			return true;
		if(rank > 0 && predicate.test(map[file][rank - 1]))
			return true;
		return false;
	}
	
	/**
	 * Checks whether or not the game has ended.
	 * 
	 * @return true if the game is over, false otherwise
	 */
	public boolean over() {
		return over;
	}
	
	/**
	 * Tests whether or not a given square has been visited.
	 * 
	 * @param file the file (a=0, b=1, ...) of the given square
	 * @param rank the rank (0=1, 1=2, ...) of the given square
	 * @return true if the square has been visited, false otherwise
	 */
	public boolean visited(int file, int rank) {
		return visited[file][rank];
	}
	
	/**
	 * Tests whether or not a given square contains the player.
	 * 
	 * @param file the file (a=0, b=1, ...) of the given square
	 * @param rank the rank (0=1, 1=2, ...) of the given square
	 * @return true if the square contains the player, false otherwise
	 */
	public boolean player(int file, int rank) {
		return file == playerFile && rank == playerRank;
	}
	
	/**
	 * Tests whether or not a given square contains the wumpus.
	 * 
	 * @param file the file (a=0, b=1, ...) of the given square
	 * @param rank the rank (0=1, 1=2, ...) of the given square
	 * @return true if the square contains the wumpus, false otherwise
	 */
	public boolean wumpus(int file, int rank) {
		return visited(file, rank) && map[file][rank] == Square.WUMPUS;
	}
	
	/**
	 * Tests whether or not a given square has a stench.
	 * 
	 * @param file the file (a=0, b=1, ...) of the given square
	 * @param rank the rank (0=1, 1=2, ...) of the given square
	 * @return true if the square has a stench, false otherwise
	 */
	public boolean stench(int file, int rank) {
		return visited(file, rank) && adjacent(file, rank, (s) -> { return s == Square.WUMPUS; });
	}
	
	/**
	 * Tests whether or not a given square contains a pit.
	 * 
	 * @param file the file (a=0, b=1, ...) of the given square
	 * @param rank the rank (0=1, 1=2, ...) of the given square
	 * @return true if the square contains a pit, false otherwise
	 */
	public boolean pit(int file, int rank) {
		return visited(file, rank) && map[file][rank] == Square.PIT;
	}
	
	/**
	 * Tests whether or not a given square has a breeze.
	 * 
	 * @param file the file (a=0, b=1, ...) of the given square
	 * @param rank the rank (0=1, 1=2, ...) of the given square
	 * @return true if the square has a breeze, false otherwise
	 */
	public boolean breeze(int file, int rank) {
		return visited(file, rank) && adjacent(file, rank, (s) -> { return s == Square.PIT; });
	}
	
	/**
	 * Tests whether or not a given square glitters.
	 * 
	 * @param file the file (a=0, b=1, ...) of the given square
	 * @param rank the rank (0=1, 1=2, ...) of the given square
	 * @return true if the square glitters, false otherwise
	 */
	public boolean glitter(int file, int rank) {
		return visited(file, rank) && map[file][rank] == Square.GOLD;
	}
	
	/**
	 * Changes the game state by having the player taken a single given action.
	 * 
	 * @param action the action to take
	 * @return a {@link Result} object describing the results of the action
	 */
	Result act(Action action) {
		Result result = Result.NOTHING;
		if(action == Action.GRAB && glitter(playerFile, playerRank)) {
			hasGold = true;
			result = Result.GRAB;
		}
		else if(action == Action.RIGHT && playerFile < 3) {
			playerFile++;
			result = null;
		}
		else if(action == Action.LEFT && playerFile > 0) {
			playerFile--;
			result = null;
		}
		else if(action == Action.UP && playerRank < 3) {
			playerRank++;
			result = null;
		}
		else if(action == Action.DOWN && playerRank > 0) {
			playerRank--;
			result = null;
		}
		if(result == null) {
			visited[playerFile][playerRank] = true;
			if(playerFile == 0 && playerRank == 0 && hasGold) {
				end();
				result = Result.WIN;
			}
			else if(wumpus(playerFile, playerRank)) {
				end();
				result = Result.EAT;
			}
			else if(pit(playerFile, playerRank)) {
				end();
				result = Result.FALL;
			}
			else
				result = new Result(stench(playerFile, playerRank), breeze(playerFile, playerRank), glitter(playerFile, playerRank));
		}
		return result;
	}
	
	/**
	 * Marks the game as over and reveals the map.
	 */
	private final void end() {
		over = true;
		for(int rank=3; rank>=0; rank--)
			for(int file=0; file<4; file++)
				visited[file][rank] = true;
	}
}
